# Copyright 2020 Amazon Web Services, Inc. or its affiliates. All Rights Reserved.  
# This AWS Content is provided subject to the terms of the AWS Customer Agreement available at  
# http://aws.amazon.com/agreement or other written agreement between Customer and either
# Amazon Web Services, Inc. or Amazon Web Services EMEA SARL or both.

from __future__ import print_function
from crhelper import CfnResource
import requests
import json
import boto3
import base64
import os
from botocore.exceptions import ClientError
import logging # Native Python library don't add to requirements
requests.packages.urllib3.disable_warnings() #Disable SSL warnings in requests

## Pre-Requisites to running this within a VPC
# SecretsManager endpoint needs to be added to the SSM Endpoint CFN
# SSM endpoint
# STS endpoint
# S3 gateway

logger = logging.getLogger(__name__)

# Initialise the helper, all inputs are optional, this example shows the defaults
helper = CfnResource(json_logging=False, log_level='INFO', boto_level='CRITICAL', sleep_on_delete=120)

## Get the Secret Function
def get_secret(*args): # We may either have 2 or 4 arguments depending if remote or local
    secret_name = args[0]
    region_name = args[1]
    scope = args[2]
    logger.debug("In the get_secret function")
    
    if scope == "remote":
        logger.debug("This is a remote Secret Key")
        logger.debug("We are going to assume Role "+ args[3])
        sts_connection = boto3.client(service_name='sts',endpoint_url="https://sts."+region_name+".amazonaws.com",region_name=region_name)
        logger.debug("sts_connection variable")
        acct_b = sts_connection.assume_role(
            RoleArn=args[3],
            RoleSessionName="cross_acct_lambda"
        )
        logger.debug("We have assumed the role")
        ACCESS_KEY = acct_b['Credentials']['AccessKeyId']
        SECRET_KEY = acct_b['Credentials']['SecretAccessKey']
        SESSION_TOKEN = acct_b['Credentials']['SessionToken']
    
        # create service client using the assumed role credentials, e.g. S3
        session = boto3.session.Session()
        client = session.client(
            service_name='secretsmanager',
            region_name=region_name,
            aws_access_key_id=ACCESS_KEY,
            aws_secret_access_key=SECRET_KEY,
            aws_session_token=SESSION_TOKEN
        )
        logger.debug("We have created the client object")
    else: 
        # Create a Secrets Manager client
        logger.debug("This is a local Secret Key")
        session = boto3.session.Session()
        client = session.client(
            service_name='secretsmanager',
            region_name=region_name
        )

    # In this sample we only handle the specific exceptions for the 'GetSecretValue' API.
    # See https://docs.aws.amazon.com/secretsmanager/latest/apireference/API_GetSecretValue.html
    # We rethrow the exception by default.

    try:
        get_secret_value_response = client.get_secret_value(
            SecretId=secret_name
        )
    except ClientError as e:
        if e.response['Error']['Code'] == 'DecryptionFailureException':
            # Deal with the exception here, and/or rethrow at your discretion.
            raise ValueError("Secrets Manager can't decrypt the protected secret text using the provided KMS key.")
        elif e.response['Error']['Code'] == 'InternalServiceErrorException':
            raise ValueError ("An error occurred on the server side.")
        elif e.response['Error']['Code'] == 'InvalidParameterException':
            raise ValueError("You provided an invalid value for a parameter.")
        elif e.response['Error']['Code'] == 'InvalidRequestException':
            raise ValueError("You provided a parameter value that is not valid for the current state of the resource.")
        elif e.response['Error']['Code'] == 'ResourceNotFoundException':
            raise ValueError("We can't find the resource that you asked for.")
        else:
            raise ValueError("Unknown Error: "+e.response['Error']['Code'])
    else:
        # Decrypts secret using the associated KMS CMK.
        # Depending on whether the secret is a string or binary, one of these fields will be populated.
        logger.debug("No Exception")
        if 'SecretString' in get_secret_value_response:
            secret = get_secret_value_response['SecretString']
            logger.debug("Returning your secret in string form")
            return(secret)
        else:
            decoded_binary_secret = base64.b64decode(get_secret_value_response['SecretBinary'])
            logger.debug("Returning your secret in binary form")
            return(secret)

def initaliseEnvironment(event, _):
## Lets get the Infoblox environment variables
    #InfobloxMember = os.environ['InfobloxMember']
    WAPIUrl =  os.environ['WAPIUrl']
    InfobloxUser_SecretKey = os.environ['InfobloxUserSecretKey']
    InfobloxPassword_SecretKey = os.environ['InfobloxPasswordSecretKey']
    Region = os.environ.get('AWS_REGION')

    # Version 2 is all local
    InfobloxUser = get_secret(InfobloxUser_SecretKey,Region,"local")
    InfobloxPassword = get_secret(InfobloxPassword_SecretKey,Region,"local")

    # Stack Metadata
    StackId = (event['StackId'])
    LogicalResourceId = (event['LogicalResourceId'])

    # API Specific Values passed via the Custom Resource
    DNSName = (event['ResourceProperties']['DNSName'])
    DNSType = (event['ResourceProperties']['DNSType']).lower()
    DNSValue = (event['ResourceProperties']['DNSValue'])

    # Static Variables
    headers = { 'content-type': "application/json" }

    # The StackID+LogicalResourceId provides a unique reference
    stackuniqref = (StackId+":"+LogicalResourceId)

    logger.debug("Infoblox WAPI URL: "+WAPIUrl)
    logger.debug("Region: "+Region)
    logger.debug("Infoblox User Secret ARN: "+InfobloxUser_SecretKey)   
    logger.debug("Infoblox Password Secret ARN: "+InfobloxPassword_SecretKey)
    logger.debug("DNSName: "+DNSName)
    logger.debug("DNSType: "+DNSType)
    logger.debug("DNSValue: "+DNSValue)
    logger.debug("StackId: "+StackId)
    logger.debug("stackuniqref: "+stackuniqref)
    return WAPIUrl, InfobloxUser, InfobloxPassword, DNSName, DNSType, DNSValue, headers, stackuniqref   

@helper.create
def CreateDNSRecord(event, _):

    # Initalise the environment
    WAPIUrl, InfobloxUser, InfobloxPassword, DNSName, DNSType, DNSValue, headers, stackuniqref = initaliseEnvironment(event, _)


    # Open the Session
    wapi_session = requests.session()

    # We need to store the stack unique reference into Infoblox so we can use it to delete resources later,
    # also useful for the Infoblox Administrator as they can use it to identify the stack and the resource that created the object.
    # Create the Attribute if it doesn't already exist
    payload='{"name": "stackuniqref","type": "STRING"}'
    response = requests.post( WAPIUrl+"/extensibleattributedef?_return_fields%2B=name,type", headers=headers, data=payload, auth=(InfobloxUser, InfobloxPassword), verify=False)
    logger.info("Create EA Status Code = "+str(response.status_code))
    records = json.loads(response.text)

    if int(response.status_code) == 200 or int(response.status_code) == 201:
        logger.info("Extensible Attribute was successfully created")
        logger.debug(response.text)
    elif 'Duplicate' in records['text']:
        logger.info("Extensible Attribute already exists")
        logger.debug(records['text'])
    else:
        raise ValueError("Cannot add Extensible Attribute " +response.text)   
    if DNSType == "arecord":
        logger.info("Creating an A-Record")
        ARecordPayload = '{ "name":"'+DNSName+'", "ipv4addr":"'+DNSValue+'" }'
        CreateRecord_response = wapi_session.post(WAPIUrl+"/record:a?_return_fields%2B=name,ipv4addr&_return_as_object=1", data=ARecordPayload, headers=headers, auth=(InfobloxUser, InfobloxPassword), verify=False)
        logger.info("Create Record Status Code = "+str(CreateRecord_response.status_code))
        logger.info("Create Record Output = "+CreateRecord_response.text)
    elif DNSType == "cname":
        CNAMERecordPayload = '{ "name":"'+DNSName+'", "canonical":"'+DNSValue+'" }'
        CreateRecord_response = wapi_session.post(WAPIUrl+"/record:cname?_return_fields%2B=name,canonical&_return_as_object=1", data=CNAMERecordPayload, headers=headers, auth=(InfobloxUser, InfobloxPassword), verify=False)
    else:
        raise ValueError("Record Type "+DNSType+" not supported")

    # To return an error to cloudformation you raise an exception:
    if int(CreateRecord_response.status_code) != 200 and int(CreateRecord_response.status_code) != 201:
        raise ValueError("Failed to Create the "+DNSType+"\n" +CreateRecord_response.text)

    logger.info("Record has been created, next it to add an extensible attribute in Infoblox for it so we can identify it, update and/or delete it")
    # We provide a unique value to each record to ensure that only if you know the stackid and reference you can delete a record
    # (and more importantly you cannot update/delete a record you didn't add)

    records = json.loads(CreateRecord_response.text)
    logger.info("Adding the Unique Extensible Identifier for this record")
    EARecordPayload = '{"extattrs":{"stackuniqref": {"value":"'+stackuniqref+'"}}}'
    AddEA_response = wapi_session.put(WAPIUrl+"/"+records['result']['_ref'], data=EARecordPayload, headers=headers, auth=(InfobloxUser, InfobloxPassword), verify=False)
    logger.debug("Create Record Status Code = "+str(AddEA_response.status_code))
    logger.debug("Create Record Output = "+AddEA_response.text)

    # To return an error to cloudformation you raise an exception:
    if int(AddEA_response.status_code) != 200 and int(AddEA_response.status_code) != 201:
        raise ValueError("Added the DNS Record but failed to add the Extensible Attribute\n" +AddEA_response.text)

    helper.Data['infobloxref'] = (records['result']['_ref'])
#End of Function

@helper.update
def update(event, _):
    DeleteDNSRecord(event, _)
    CreateDNSRecord(event, _)


@helper.delete
def DeleteDNSRecord(event, _):

    # Initalise the environment
    WAPIUrl, InfobloxUser, InfobloxPassword, DNSName, DNSType, DNSValue, headers, stackuniqref = initaliseEnvironment(event, _)

    ## First we need to find the record using the Unique Stack ID
    # Open the Session
    wapi_session = requests.session()
    RecordPayload = '{ "*stackuniqref":"'+stackuniqref+'"}'
    if DNSType == "arecord":

        FindRecord_response = wapi_session.get(WAPIUrl+"/record:a?", data=RecordPayload, headers=headers, auth=(InfobloxUser, InfobloxPassword), verify=False)
        # To return an error to cloudformation you raise an exception:
        if int(FindRecord_response.status_code) != 200 and int(FindRecord_response.status_code) != 201:
            raise ValueError("API Call to find A Record failed \n" +FindRecord_response.text)
    elif DNSType == "cname":
        FindRecord_response = wapi_session.get(WAPIUrl+"/record:cname?", data=RecordPayload, headers=headers, auth=(InfobloxUser, InfobloxPassword), verify=False)
        # To return an error to cloudformation you raise an exception:
        if int(FindRecord_response.status_code) != 200 and int(FindRecord_response.status_code) != 201:
            raise ValueError("API Call to find the CNAME Record failed \n" +FindRecord_response.text)
    else:
        raise ValueError("Record Type "+DNSType+" not supported") 

    #Taking the output from the API request, lets check whether we get the reference from Infoblox
    records = json.loads(FindRecord_response.text)
    if not records:
        raise ValueError("Cannot find the reference for record "+DNSName+" in Infoblox, does it exist?") 
    RecordRef=(records[0]['_ref'])

    # ## Using the reference delete the record
    DeleteRecord_response = wapi_session.delete(WAPIUrl+"/"+RecordRef, headers=headers, auth=(InfobloxUser, InfobloxPassword), verify=False)

    # # To return an error to cloudformation you raise an exception:
    if int(DeleteRecord_response.status_code) != 200 and int(DeleteRecord_response.status_code) != 201:
        raise ValueError("Failed to Delete the A Record \n" +DeleteRecord_response.text)
    print("No errors, so record deleted")

def lambda_handler(event, context):
    #SNS events contain a wrapper around the Lambda event. Unpack the
    #Lambda event from SNS. Not needed if you’re calling Lambda directly.
    if 'Records' in event:
        logger.info("This is a SNS Event")
        logger.debug("SNS Event: " + json.dumps(event))
        event = json.loads(event['Records'][0]['Sns']['Message'])        
    else:
        logger.info("This is a Cloudformation Event")
        event = json.loads(event['Records'][0]['Sns']['Message'])            
        logger.debug("Lambda Event: " + json.dumps(event))
    helper(event, context)