/*
 * Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this
 * software and associated documentation files (the "Software"), to deal in the Software
 * without restriction, including without limitation the rights to use, copy, modify,
 * merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.amazon.aws.tokenVendingLayer;

import com.amazon.aws.tokenVendingLayer.exceptions.PolicyAssumptionException;
import com.amazon.aws.tokenVendingLayer.policy.DeclarativePolicyGenerator;
import com.amazon.aws.tokenVendingLayer.policy.PolicyGenerator;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.auth.credentials.EnvironmentVariableCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.http.urlconnection.UrlConnectionHttpClient;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.sts.StsClient;
import software.amazon.awssdk.services.sts.model.AssumeRoleResponse;
import software.amazon.awssdk.services.sts.model.Credentials;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * Lambda function entry point. You can change to use other pojo type or implement
 * a different RequestHandler.
 *
 * @see <a href=https://docs.aws.amazon.com/lambda/latest/dg/java-handler.html>Lambda Java Handler</a> for more information
 */
public class TokenVendingMachine {

    private static final Logger LOGGER = LoggerFactory.getLogger(TokenVendingMachine.class);
    private static final ObjectMapper MAPPER = new ObjectMapper();

    final Region region = Region.of(System.getenv("AWS_REGION"));
    final String role = System.getenv("ROLE");
    final int durationSeconds = 900;
    private final StsClient sts = StsClient.builder()
            .region(region)
            .httpClientBuilder(UrlConnectionHttpClient.builder())
            .credentialsProvider(EnvironmentVariableCredentialsProvider.create())
            .build();

    public AwsCredentialsProvider vendToken(String tenant, String bucket) {
        PolicyGenerator policyGenerator = DeclarativePolicyGenerator.generator()
                .tenant(tenant)
                .s3FolderPerTenant(bucket);
        String scopedPolicy = policyGenerator.generatePolicy();
        LOGGER.info(scopedPolicy);
        return getCredentialsForTenant(scopedPolicy, tenant, role);
    }

    private AwsCredentialsProvider getCredentialsForTenant(String scopedPolicy, String tenant, String role) {

        StaticCredentialsProvider credentialsProvider;
        Credentials scopedCredentials;

        if (scopedPolicy == null || scopedPolicy.trim().isEmpty()) {
            LOGGER.info("TokenVendingMachine::Attempting to assumeRole with empty policy, should not happen!");
            throw new PolicyAssumptionException("Missing or empty policy, cannot allow access.");
        }
        try {
            AssumeRoleResponse assumeRoleResponse = sts
                    .assumeRole(assumeRoleReq -> assumeRoleReq
                            .durationSeconds(durationSeconds)
                            .policy(scopedPolicy)
                            .roleArn(role)
                            .roleSessionName(tenant)
                    );
            scopedCredentials = assumeRoleResponse.credentials();
            credentialsProvider = StaticCredentialsProvider.create(
                    AwsSessionCredentials.create(scopedCredentials.accessKeyId(), scopedCredentials.secretAccessKey(), scopedCredentials.sessionToken())
            );
        } catch (SdkServiceException stsError) {
            LOGGER.error("STS::AssumeRole", stsError);
            throw new RuntimeException(stsError);
        }

        return credentialsProvider;
    }
}
