function Get-ProxySettings {
    <#
    .SYNOPSIS
    Validates that http_proxy or https_proxy is set .
    .DESCRIPTION
    Validates that http_proxy or https_proxy is set.  Returns an object with result and friendly description
    .PARAMETER Enabled
    Should this test be run? True/False.
    .PARAMETER Label
    Friendly name for this Test.
    .EXAMPLE
    Get-ProxySettings -Enabled $True 

    Description
    -----------
    Validates that http_proxy or https_proxy is set.  Returns an object with result and friendly description
    #>

    [CmdletBinding()]

    Param
    (
        [Parameter(Mandatory=$True, HelpMessage = 'Should this test be run? True/False')]
        [boolean] $Enabled,

        [Parameter(HelpMessage = 'Friendly name for this Test')]
        [string] $Label = "Proxy Enabled"
    )

    # Set Default Parameters in Hashtable
    $DefaultTable = New-Object HashTable
    $DefaultTable.Add('Enabled', $True)

    # Compare JSON Parameters to Default and set as required
    $JSONParms = $PSBoundParameters
    $FinalParams = Compare-Parameters $JSONParms $DefaultTable

    If ($FinalParams.Enabled) {
        Write-Verbose "Info - Attempting to get http(s)_proxy setting "
        $TestResult = "Pass"

        if ( (Test-Path -Path 'env:https_proxy') -or (Test-Path -Path 'env:http_proxy')) {
            $TestResult = "Fail"
        }

        $keys = @(
                "HKU:\.Default\SOFTWARE\Microsoft\Windows\CurrentVersion\Internet Settings",
                "HKCU:\Software\Microsoft\Windows\CurrentVersion\Internet Settings",
                "HKU:\S-1-5-18\SOFTWARE\Microsoft\Windows\CurrentVersion\Internet Settings"
                "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Internet Settings"
                )

        foreach($key in $keys){
            try {
                If((Get-Item -Path $key -ErrorAction Stop).GetValue("ProxyEnable")  -gt 0)
                    {
                        $TestResult="Fail"
                        break
                    }
                }
            catch{
                    continue
            }
        }

        $WinHttpProxy = Invoke-ExternalCommand -LiteralPath c:\Windows\System32\netsh.exe winhttp show proxy

        
        if (($WinHttpProxy | Select-String -Pattern "no proxy") -eq $null)
            {
            $TestResult="Fail"
           
        }

    }
    else {
        $TestResult = "Skipped"
    }

    
    
    if($TestResult -eq "Pass") {
        $TestDescription = "Validates that Proxy is not set on the instance"
    } else
    {
        $TestDescription = "Please make sure the proxy settings is properly configured. More info: https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-install-ssm-proxy.html "
    }

    $ResultObject = [PSCustomObject] @{'Label' = $Label; 'Result' = $TestResult; 'Enforcement' = 'Recommended'; 'Configuration' = $FinalParams.Configuration; 'Description' = $TestDescription }

    return $ResultObject
}

