function Confirm-SysprepRegistryValues {
     <#
     .SYNOPSIS
     Validates that the instance has the correct registry values for multiple sysprep to be successful.
     .DESCRIPTION
     Validates that the instance has the correct registry values for multiple sysprep to be successful.
     .PARAMETER Enabled
     Should this test be run? True/False.
     .PARAMETER Label
     Friendly name for this Test.
     .EXAMPLE
     Confirm-SysprepRegistryValues -Enabled $True
 
     Description
     -----------
     Validates that the instance has the correct registry values for multiple sysprep to be successful.
     Running sysprep for more than 3 times require a few registry values to be set. Below are the related links :
       https://docs.microsoft.com/en-US/troubleshoot/windows-client/deployment/error-occurs-when-running-sysprep-generalize
       https://www.repairwin.com/how-to-sysprep-windows-more-than-three-3-times-rearm/
       https://docs.microsoft.com/en-us/windows-hardware/manufacture/desktop/windows-setup-states
     #>
 
     [CmdletBinding()]

	# Set Default Parameters in Hashtable
	$DefaultTable = New-Object HashTable
	$DefaultTable.Add('Enabled', $True)
	 
	# Compare Passed Parameters to Default and set as required
	$JSONParms = $PSBoundParameters
	$FinalParams = Compare-Parameters $JSONParms $DefaultTable
	 
	if ($FinalParams.Enabled) {
	    Write-Verbose "Info - Attempting to get the required registry values"

	    $sysprepStatusPath = 'HKLM:\SYSTEM\Setup\Status\SysprepStatus'
	    $softwareProtectionPlatformPath = 'HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\SoftwareProtectionPlatform'
	    $imageStatePath = 'HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Setup\State'
	    $streamProviderPath = 'HKLM:\SOFTWARE\Microsoft\Windows\StreamProvider'

	    # Set Defaults before Tests
	    $Script:TestResult = "Pass"
	    $TestDescription = "Validates that the instance has the correct registry values."
	    $Script:FailDescription = New-Object HashTable

	    if (Test-Path -Path $sysprepStatusPath) {
            $message = $sysprepStatusPath + "\CleanupState should be set to 2"
            sysprep_key_state -path $sysprepStatusPath -registryName "CleanupState" -expectedValue 2 -message $message
            
            $message = $sysprepStatusPath + "\GeneralizationState should be set to 7"
            sysprep_key_state -path $sysprepStatusPath -registryName "GeneralizationState" -expectedValue 7 -message $message
        }

        if (Test-Path -Path $softwareProtectionPlatformPath) {
            $message = $softwareProtectionPlatformPath + "\SkipRearm should be set to 1"
            sysprep_key_state -path $softwareProtectionPlatformPath -registryName "SkipRearm" -expectedValue 1 -message $message
        }

        if (Test-Path -Path $imageStatePath) {
            $message = $imageStatePath + "\ImageState should be set to IMAGE_STATE_COMPLETE"
            sysprep_key_state -path  $imageStatePath -registryName "ImageState" -expectedValue 'IMAGE_STATE_COMPLETE' -message $message
	    }
	 
	    if (Test-Path -Path $streamProviderPath) {
            $message = $streamProviderPath + "\LastFullPayloadTime should be set to 0"
            sysprep_key_state -path $streamProviderPath -registryName "LastFullPayloadTime" -expectedValue 0 -message $message
	    }
	 
	    if ( $TestResult -eq "Fail") {
	        $TestDescription = $FailDescription.Values | Out-String
	    }
	}
    else {
        $TestResult = "Skipped"
        $TestDescription = "Validates that the instance has the correct registry values."
    }
     
    $ResultObject = [PSCustomObject] @{'Label' = $Label; 'Result' = $TestResult; 'Enforcement' = 'Recommended'; 'Configuration' = $FinalParams.Configuration; 'Description' = $TestDescription }
    Write-Host $ResultObject
    return $ResultObject
}

function sysprep_key_state{
    Param(
    [string] $path, 
    [string] $registryName, 
    [string] $expectedValue, 
    [string] $message
    )

    $currentValue = (Get-ItemProperty -Path $path -ErrorAction Stop).$registryName
    if ($currentValue -ne $expectedValue){
        $Script:FailDescription.$registryName = $message
        $Script:TestResult = "Fail"
    }
}
     
Confirm-SysprepRegistryValues -Enabled $True