function Get-Endpoints {
	     <#
	     .SYNOPSIS
	     Checks endpoint of ssm , s3 and cloudformation 
	     .DESCRIPTION
	     Checks endpoint of ssm , s3 and cloudformation . Returns an object with result and friendly description
	     .PARAMETER Enabled
	     Should this test be run? True/False.
	     .PARAMETER Label
	     Friendly name for this Test.
	     .PARAMETER EndpointList
	     Failed endpoints 
	     .EXAMPLE
	     Get-Endpoints -Enabled $True 
	 
	     Description
	     -----------
	     Checks endpoint of ssm , s3 and cloudformation . Returns an object with result and friendly description
	     #>
	 
	     [CmdletBinding()]
	 
	     Param
	     (
	         [Parameter(Mandatory = $True, HelpMessage = 'Should this test be run? True/False')]
	         [boolean] $Enabled,
	 
	         [Parameter(HelpMessage = 'Friendly name for this Test')]
	         [string] $Label = "Check Endpoints"
	 
	         
	     )
	 
	     # Set Default Parameters in Hashtable
	     $DefaultTable = New-Object HashTable
	     $DefaultTable.Add('Enabled', $True)
	     
	 
	     # Compare Passed Parameters to Default and set as required
	     $JSONParms = $PSBoundParameters
	     $FinalParams = Compare-Parameters $JSONParms $DefaultTable
	 
	     If ($FinalParams.Enabled) {
	         Write-Verbose "Info - Checking endpoints"
	         
	         $Endpoints = @("s3","ssm", "cloudformation")
	         $Description = ""
	         $TestResult = "Pass"
	         $webClient = New-Object System.Net.WebClient;
	         try {
	 
	             $webClient = New-Object -TypeName System.Net.WebClient;
	             $Region=$webClient.DownloadString("http://169.254.169.254/latest/meta-data/placement/availability-zone")
	             $Region=$Region.Substring(0,$Region.Length-1)
	             [int32]$TimeOutSeconds = 10000
	             ForEach ($Endpoint in $Endpoints) {
	                     try {
	                            $tcp = New-Object System.Net.Sockets.TcpClient
		                        $connection = $tcp.BeginConnect("$Endpoint.$Region.amazonaws.com", 443, $null, $null)
		                        $connection.AsyncWaitHandle.WaitOne($TimeOutSeconds,$false)  | Out-Null 
		                        if($tcp.Connected -eq $false) {
                                    $TestResult = "Fail"
			                        $Description = $Description + "$Endpoint.$Region.amazonaws.com is not reachable`n"
		                        }
	                         }
	                     catch {
	                             $TestResult = "Fail"
	                             $Description = $Description + "Unknown error while connection to $Endpoint.$Region.amazonaws.com `n"
	                         }
	                 }
	             }
	             catch {
	                     $TestResult = "Fail"
	                     $Description = "Unable to fetch region and endpoints. Check the persistent routes for 169.254.169.254.`nMore info: https://aws.amazon.com/premiumsupport/knowledge-center/waiting-for-metadata/"
	                 }
	             if ($TestResult -eq "Pass") {
	                 $Description = "Validates if all AMS endpoints are reachable by the instance."
	             }
	         }
	    
	     else {
	         $TestResult = "Skipped"
	     }
	 
	 
	     $ResultObject = [PSCustomObject] @{'Label' = $Label; 'Result' = $TestResult; 'Enforcement' = 'Required'; 'Configuration' = $FinalParams.Configuration; 'Description' = $Description }
	     return $ResultObject
	 }
