function Confirm-RDSLicenseGracePeriod {
     <#
     .SYNOPSIS
     Validates that the instance has the Remote Desktop License Servers available to provide a license.
     .DESCRIPTION
     Validates that the instance has the Remote Desktop License Servers available to provide a license.
     .PARAMETER Enabled
     Should this test be run? True/False.
     .PARAMETER Label
     Friendly name for this Test.
     .EXAMPLE
     Confirm-RDSLicenseGracePeriod -Enabled $True
 
     Description
     -----------
     Validates that the instance has the Remote Desktop License Servers available to provide a license.
     #>
 
     [CmdletBinding()]

     # Set Default Parameters in Hashtable
     $DefaultTable = New-Object HashTable
     $DefaultTable.Add('Enabled', $True)

     # Compare Passed Parameters to Default and set as required
     $JSONParms = $PSBoundParameters
     $FinalParams = Compare-Parameters $JSONParms $DefaultTable

     if ($FinalParams.Enabled) {
          Write-Verbose "Info - Attempting to get the grace period for the RDS license"

          # Set Defaults before Tests
          $TestResult = "Pass"
          $TestDescription = "Validates that the instance has the Remote Desktop License Servers available to provide a license."
          
          $GracePeriodRegistryPath = 'HKLM:\SYSTEM\CurrentControlSet\Control\Terminal Server\RCM\GracePeriod'

          if (Test-Path -Path $GracePeriodRegistryPath) {
               $GracePeriod = (Invoke-WmiMethod -PATH (Get-WmiObject -namespace root\cimv2\terminalservices -class win32_terminalservicesetting).__PATH -name GetGracePeriodDays).daysleft
               if ($GracePeriod) {
                    $TestResult = "Warning"
                    $TestDescription = "WARNING: The machine is currently in grace period for the next $($GracePeriod) days. Kindly check the machine's RDS license configuration to avoid losing access post the grace period expires. Please contact server administrator for further assistance. To know more, please check: https://docs.microsoft.com/en-us/troubleshoot/windows-server/remote/cannot-connect-rds-no-license-server#refresh-th%5B%E2%80%A6%5Dgistry-keys"
               }
          }
          else {
               $TestResult = "Pass"
          }
     }
     else {
          $TestResult = "Skipped"
          $TestDescription = "Validates that the instance has the Remote Desktop License Servers available to provide a license."
     }

     $ResultObject = [PSCustomObject] @{'Label' = $Label; 'Result' = $TestResult; 'Enforcement' = 'Recommended'; 'Configuration' = $FinalParams.Configuration; 'Description' = $TestDescription }
     return $ResultObject
}
