import distro
import grp
from pre_wigs_validation.enums import ValidationEnforcement, ValidationResult
from pre_wigs_validation.instance import ValidationInstance
from pre_wigs_validation.dataclasses import ValidationOutput
from pre_wigs_validation.utils import check_validation_config


class NSCDGroupCheck:
    """Validate that group ID 28 is not in use by any service other than nscd on AmazonLinux2, CentOS, RHEL"""

    validation = "NSCD Group Check"
    enforcement = ValidationEnforcement.REQUIRED

    @classmethod
    def validate(
        cls, *, enabled: bool = True, instance: ValidationInstance
    ) -> ValidationOutput:

        """
        Parameters:
        enabled (bool): whether or not to run this validation function
        instance (ValidationInstance): the instance object being validated

        Returns:
        ValidationOutput: output of validation
        """

        if not enabled:
            return ValidationOutput(
                validation=cls.validation,
                result=ValidationResult.NOT_RUN,
                enforcement=cls.enforcement,
            )


        config = check_validation_config(
            default_params=cls.validate.__kwdefaults__, local_params=locals()
        )

        if cls.is_impacted_operating_system(instance):
            try:
                group= grp.getgrgid(28)
            except KeyError:
                pass_message = "Group ID 28 is currently not in use."
                return ValidationOutput(
                    validation=cls.validation,
                    result=ValidationResult.PASS,
                    enforcement=cls.enforcement,
                    config=config,
                    message=pass_message,
                )
            else:
                if(group.gr_name == 'nscd'):
                    pass_message = "Group ID 28 is already in use by nscd. No action required"
                    return ValidationOutput(
                        validation=cls.validation,
                        result=ValidationResult.PASS,
                        enforcement=cls.enforcement,
                        config=config,
                        message=pass_message,
                    )
                else:
                    group_name = group.gr_name
                    fail_message = f"Group ID 28 is currently in use by {group_name}. Please use a different GID for this group as group ID 28 is reserved for nscd."
                    return ValidationOutput(
                        validation=cls.validation,
                        result=ValidationResult.FAIL,
                        enforcement=cls.enforcement,
                        config=config,
                        message=fail_message,
                    )

        else:
            pass_message = f"Operating system not impacted"
            return ValidationOutput(
                validation=cls.validation,
                result=ValidationResult.PASS,
                enforcement=cls.enforcement,
                config=config,
                message=pass_message,
            )


    @classmethod
    def is_impacted_operating_system(cls, instance: ValidationInstance):

        dist = instance.distribution

        is_impacted = False

        if dist == "centos":
            is_impacted = True
        elif dist == "rhel":
            is_impacted = True
        elif dist == "amzn":
            is_impacted = True

        return is_impacted
