function Add-FSxADPermissions {
<#
.SYNOPSIS
    Add required Active Directory permissions for Amazon FSx service account integration

.DESCRIPTION
    Add required permissions to Amazon FSx service account that automates requirements
    as documented in "Delegating Privileges to Your Amazon FSx Service Account"
    This script is self-contained and can be reviewed and executed by your domain administrator
    on an instance joined to the domain.

.EXAMPLE
    # OrganizationalUnit not provided, will default to Computers container for domain
    Add-FSxADPermissions -UserName 'fsxServiceUser'

.EXAMPLE
    # Supports -WhatIf parameter to both list existing permissions and see what
    # permissions would be added without actually adding them.
    Add-FSxADPermissions -UserName 'fsxServiceUser' -WhatIf

.EXAMPLE
    Add-FSxADPermissions -UserName 'fsxServiceUser' -OrganizationalUnit 'OU=Amazon FSx,OU=test-ad,DC=test-ad,DC=local'

.LINK
    Delegating Privileges to Your Amazon FSx Service Account
    https://docs.aws.amazon.com/fsx/latest/WindowsGuide/self-managed-AD-best-practices.html#connect_delegate_privileges

    Well-known security identifiers in Windows operating systems
    https://support.microsoft.com/en-us/help/243330/well-known-security-identifiers-in-windows-operating-systems

    Default AD DS schema property set
    https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-adts/177c0db5-fa12-4c31-b75a-473425ce9cca
#>
[CmdletBinding(SupportsShouldProcess)]
[OutputType([Object])]
param (
    [Parameter(Mandatory = $True,
               Position = 0)]
    [ValidateNotNullOrEmpty()]
    [string]
    $UserName, # UserName of the Amazon FSx service account

    [Parameter(Mandatory = $False,
               Position = 1)]
    [string]
    $OrganizationalUnit # Distinguished name for Organizational Unit (OU) within the domain that you want your Amazon FSx file system to join to. Defaults to Computers container for domain.
)
Process {
    $ServiceAccountIdentity = [System.Security.Principal.IdentityReference](Get-ADUser $UserName).SID
    $ServiceAccount = $ServiceAccountIdentity.Translate([System.Security.Principal.NTAccount])

    $IsDefaultComputers = $False
    If (!$OrganizationalUnit) {
        $AdDomain = Get-ADDomain
        $OrganizationalUnit = $AdDomain.ComputersContainer
        Write-Host "Using the Active Directory's Computers container because no Organizational Unit was specified: ${OrganizationalUnit}"
        $DefaultComputersName = "CN=Computers," + $AdDomain.DistinguishedName
        $IsDefaultComputers = ($OrganizationalUnit -eq $DefaultComputersName)
    }

    $DistinguishedNameFilter = "DistinguishedName -eq '$OrganizationalUnit'"
    If ($IsDefaultComputers) {
        # This is a container and not a OU
        If (!(Get-ADObject -Filter $DistinguishedNameFilter)) {
            Write-Error "Failed to access default CN=Computers container ${OrganizationalUnit}"
            Return 1
        }
    } Else {
        # Verify OU exists and is accessible
        If (!(Get-ADOrganizationalUnit -Filter $DistinguishedNameFilter)) {
            $RDNs = ($OrganizationalUnit -Split ',')
            $Name = ($RDNs[0] -Split '=')[1]
            $Path = ($RDNs[1..($RDNs.Length - 1)] -Join ',')

            $ErrorMessage  = "Unable to find an Organizational Unit (OU) with the following distinguished name: ${OrganizationalUnit}`n"
            $ErrorMessage += "To create this OU, run the following command: `n"
            $ErrorMessage += ("New-AdOrganizationalUnit -Name '${Name}' -Path '${Path}' -ProtectedFromAccidentalDeletion " + '$' + "True")

            Write-Error $ErrorMessage
            Return 1
        }
    }

    # https://docs.microsoft.com/en-us/windows/win32/adschema/c-computer
    $ComputerSchemaGuid = [guid]"bf967a86-0de6-11d0-a285-00aa003049e2"
    $Allow = [System.Security.AccessControl.AccessControlType]"Allow"
    $Deny = [System.Security.AccessControl.AccessControlType]"Deny"
    $All = [System.DirectoryServices.ActiveDirectorySecurityInheritance]"All"
    $Descendents = [System.DirectoryServices.ActiveDirectorySecurityInheritance]"Descendents"
    $None = [System.DirectoryServices.ActiveDirectorySecurityInheritance]"None"

    $TargetServiceAccountAccessRules = @{}

    # Add "Create" and "Delete" computer object permission
    $AdRights = [System.DirectoryServices.ActiveDirectoryRights]@("CreateChild", "DeleteChild")
    $AccessControlType = $Allow
    $ObjectTypeValue = $ComputerSchemaGuid
    $Inheritance = $All
    $InheritedObjectTypeValue = [System.Guid]::empty
    $CreateAndDeleteComputerObjectsACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $AdRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue
    $TargetServiceAccountAccessRules['Create and delete computer objects'] = $CreateAndDeleteComputerObjectsACR

    # Add "ListChildren" and "ReadProperties" permission
    $AdRights = [System.DirectoryServices.ActiveDirectoryRights]@("ListChildren", "ReadProperty")
    $AccessControlType = $Allow
    $ObjectTypeValue = [System.Guid]::empty
    $Inheritance = $None
    $InheritedObjectTypeValue = [System.Guid]::empty
    $ListChildrenAndReadACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $AdRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue
    $TargetServiceAccountAccessRules['List children and read property'] = $ListChildrenAndReadACR

    # Add "Read and write Account Restrictions" permission
    $AdRights = [System.DirectoryServices.ActiveDirectoryRights]@("ReadProperty", "WriteProperty")
    $AccessControlType = $Allow
    # https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-adts/177c0db5-fa12-4c31-b75a-473425ce9cca
    $ObjectTypeValue = [guid]"4c164200-20c0-11d0-a768-00aa006e0529" # Account Restrictions
    $Inheritance = $Descendents
    $InheritedObjectTypeValue = $ComputerSchemaGuid
    $ReadAndWriteAccountRestrictionsACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $AdRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue
    $TargetServiceAccountAccessRules['Read and write Account Restrictions'] = $ReadAndWriteAccountRestrictionsACR

    # Add "WriteDacl" permission
    $AdRights = [System.DirectoryServices.ActiveDirectoryRights]@("WriteDacl")
    $AccessControlType = $Allow
    $ObjectTypeValue = $ComputerSchemaGuid
    $Inheritance = $All
    $InheritedObjectTypeValue = [System.Guid]::empty
    $ModifyComputerObjectsACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $AdRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue
    $TargetServiceAccountAccessRules['Modify computer objects'] = $ModifyComputerObjectsACR

    # Add "Validated write to service principal name" permission
    $AdRights = [System.DirectoryServices.ActiveDirectoryRights]@("Self")
    $AccessControlType = $Allow
    # https://docs.microsoft.com/en-us/windows/win32/adschema/a-serviceprincipalname
    $ObjectTypeValue = [guid]"f3a64788-5306-11d1-a9c5-0000f80367c1"
    $Inheritance = $Descendents
    $InheritedObjectTypeValue = $ComputerSchemaGuid
    $ValidatedWriteToSPNACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $AdRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue
    $TargetServiceAccountAccessRules['Validated write to service principal name'] = $ValidatedWriteToSPNACR

    # Add "Validated write to DNS host name" permission
    $AdRights = [System.DirectoryServices.ActiveDirectoryRights]@("Self")
    $AccessControlType = $Allow
    # https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-adts/177c0db5-fa12-4c31-b75a-473425ce9cca
    $ObjectTypeValue = [guid]"72e39547-7b18-11d1-adef-00c04fd8d5cd" # DNS Host Name Attribute
    $Inheritance = $Descendents
    $InheritedObjectTypeValue = $ComputerSchemaGuid
    $ValidatedWriteToDNSHostNameACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $AdRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue
    $TargetServiceAccountAccessRules['Validated write to DNS host name'] = $ValidatedWriteToDNSHostNameACR

    # Add "Validated write to DNS host name" permission
    $AdRights = [System.DirectoryServices.ActiveDirectoryRights]@("ExtendedRight")
    $AccessControlType = $Allow
    # https://docs.microsoft.com/en-us/windows/win32/adschema/r-user-force-change-password
    $ObjectTypeValue = [guid]"00299570-246d-11d0-a768-00aa006e0529" # Reset Password
    $Inheritance = $Descendents
    $InheritedObjectTypeValue = $ComputerSchemaGuid
    $ResetPasswordsACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $AdRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue
    $TargetServiceAccountAccessRules['Reset Password'] = $ResetPasswordsACR

    # Add "ListChildren" permission for all objects in 'This Organization'
    # https://support.microsoft.com/en-us/help/243330/well-known-security-identifiers-in-windows-operating-systems
    $ThisOrganizationIdentity = New-Object System.Security.Principal.SecurityIdentifier("S-1-5-15") # This Organization
    $PreWin2kCompatibleAccessIdentity = New-Object System.Security.Principal.SecurityIdentifier("S-1-5-32-554") # Builtin\Pre-Windows 2000 Compatible Access
    $ListChildrenRights = [System.DirectoryServices.ActiveDirectoryRights]@("ListChildren")
    $AccessControlType = $Allow
    $ObjectTypeValue = [System.Guid]::empty
    $Inheritance = $None
    $InheritedObjectTypeValue = [System.Guid]::empty
    $ThisOrganizationListChildrenACR = New-Object System.DirectoryServices.ActiveDirectoryAccessRule $ServiceAccountIdentity, $ListChildrenRights, $AccessControlType, $ObjectTypeValue, $Inheritance, $InheritedObjectTypeValue

    $OuAdsi = [adsi]"LDAP://${OrganizationalUnit}"
    # Only read/write discretionary access-control list (DACL)
    # https://docs.microsoft.com/en-us/dotnet/api/system.directoryservices.securitymasks?view=dotnet-plat-ext-3.1
    $OuAdsi.PSBase.Options.SecurityMasks = [System.DirectoryServices.SecurityMasks]::Dacl

    $ExistingAccessRules = (Get-Acl -Path "AD:\${OrganizationalUnit}").Access
    $ExistingServiceAccountAccessRules = $ExistingAccessRules | Where-Object { $_.IdentityReference -eq $ServiceAccount }

    $TargetServiceAccountAccessRules.GetEnumerator() | ForEach-Object { $_.Key } | % {
        $AccessRule = $TargetServiceAccountAccessRules[$_]
        If ($AccessRule) {
            $ExistingMatchingRules = ($ExistingServiceAccountAccessRules |  Where-Object { ($_.ActiveDirectoryRights -eq $AccessRule.ActiveDirectoryRights) -and ($_.ObjectType.Guid -eq $AccessRule.ObjectType.Guid) })
            If ($ExistingMatchingRules.Count -eq 0) {
                $AddRuleDescription = "Allow ${UserName} to $_ on ${OrganizationalUnit}"
                If ($PSCmdlet.ShouldProcess($_)) {
                    Write-Host $AddRuleDescription
                    $AccessRule | Format-List | Out-String | Write-Host
                    $OuAdsi.PSBase.ObjectSecurity.AddAccessRule($AccessRule)
                } Else {
                    Write-Host "WhatIf: ${AddRuleDescription}"
                    $AccessRule | Format-List | Out-String | Write-Host
                }
            } Else {
                $ExistingMatchingDenyRules = ($ExistingMatchingRules | Where-Object { $_.AccessControlType -eq $Deny })
                If ($DenyRules.Count -eq 0) {
                    Write-Host "Existing service account permission found for $_"
                } Else {
                    Write-Warning "Existing deny rule found for $_ on ${OrganizationalUnit}. Please remove it."
                }
            }
        } Else {
            Write-Warning "Access rule empty for $_"
        }
    }

    $ExistingListChildrenAccessRules = $ExistingAccessRules | Where-Object { $_.ActiveDirectoryRights -eq $ListChildrenRights }
    $ExistingListChildrenDenyRules = ($ExistingListChildrenAccessRules | Where-Object { $_.AccessControlType -eq $Deny })
    If ($ExistingListChildrenDenyRules.Count -gt 0) {
        Write-Warning "Existing deny rule found for ListChildren on ${OrganizationalUnit}. Please remove it."
    } Else {
        $AcceptableIdentities = @($ThisOrganizationIdentity, $PreWin2kCompatibleAccessIdentity)
        $ExistingListChildrenAllowRules = ($ExistingListChildrenAccessRules | Where-Object { $AcceptableIdentities -Contains $_.IdentityReference })
        If ($ExistingListChildrenAllowRules.Count -gt 0) {
            Write-Host "Existing allow rule found for ListChildren on ${OrganizationalUnit}."
        } Else {
            $AddRuleDescription = "Allow 'This Organization' to ListChildren on ${OrganizationalUnit}."
            If ($PSCmdlet.ShouldProcess($_)) {
                Write-Host $AddRuleDescription
                $ThisOrganizationListChildrenACR | Format-List | Out-String | Write-Host
                $OuAdsi.PSBase.ObjectSecurity.AddAccessRule($ThisOrganizationListChildrenACR)
            } Else {
                Write-Host "WhatIf: ${AddRuleDescription}"
                $ThisOrganizationListChildrenACR | Format-List | Out-String | Write-Host
            }
        }
    }
    $OuAdsi.PSBase.CommitChanges()

    $ResultServiceAccountAccessRules = (Get-Acl -Path "AD:\${OrganizationalUnit}").Access | Where-Object { ($_.IdentityReference -eq $ServiceAccount) -and ($_.AccessControlType -eq $Allow) }

    If ($PSCmdlet.ShouldProcess($_)) {
        If ($ResultServiceAccountAccessRules.Count -ge $TargetServiceAccountAccessRules.Count) {
            Write-Host "Successfully added permissions to ${UserName} for ${OrganizationalUnit}."
        } Else {
            $CallerIdentity = whoami
            Write-Host("Failed setting permissions for ${OrganizationalUnit} to ${UserName} with ${CallerIdentity}.")
        }
    } Else {
        Write-Host "Skipped setting permissions due to WhatIf configuration"
    }
    Write-Host "Current access rules below:"
    $ResultServiceAccountAccessRules | Format-Table | Out-String | Write-Host
}
}
