/*
 * Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this
 * software and associated documentation files (the "Software"), to deal in the Software
 * without restriction, including without limitation the rights to use, copy, modify,
 * merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.amazon.aws.s3UploadSample;

import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;

import com.amazonaws.services.lambda.runtime.Context;
import com.amazonaws.services.lambda.runtime.RequestHandler;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.amazon.aws.s3UploadSample.services.S3ClientService;
import com.amazon.aws.tokenVendingLayer.TokenVendingMachine;

import java.util.UUID;
import java.util.Map;
import java.util.HashMap;

/**
 * Lambda function entry point. You can change to use other pojo type or implement
 * a different RequestHandler.
 *
 * @see <a href=https://docs.aws.amazon.com/lambda/latest/dg/java-handler.html>Lambda Java Handler</a> for more information
 */
public class App implements RequestHandler<Map<String,String>, String> {
    private static final Logger LOGGER = LoggerFactory.getLogger(App.class);

    @Override
    public String handleRequest(Map<String,String> event, Context context) {

        String bucket = System.getenv("S3_BUCKET");
        String json = "{\"value\": \"test\"}";
        String tenant = event.get("tenant");
        String key = UUID.randomUUID() + ".json";

        TokenVendingMachine tokenVendingMachine = new TokenVendingMachine();
        AwsCredentialsProvider tenantCredentials = tokenVendingMachine.vendToken(tenant, bucket);

        Map<String, String> map = new HashMap<>();
        map.put("tenant", tenant);

        try {
            S3ClientService s3ClientService = new S3ClientService(bucket, tenantCredentials);
            s3ClientService.upload(json, tenant + "/" + key);

            ListObjectsV2Response listResponse = s3ClientService.listObjects(tenant);
            map.put("S3 documents visible", Integer.toString(listResponse.contents().size()));
        } catch (SdkServiceException e) {
            LOGGER.error("Issue calling the services ", e);
            map.put("S3Error", e.getMessage());
        }
        return "Function executed successfully";
    }
}
