import av
import queue
import threading
import time

MAX_CONTAINER_OPEN_WAIT_TIME = 15
MIN_CONTAINER_OPEN_WAIT_TIME = 5

class PyAvStream:
    def __init__(self, rtsp_url, max_buffer_size = 20, max_missed_frames = 10, width=800, height=600):
        print("PyAV RTSP Stream object init.")
        self.initialized = False
        self.container_open = False
        self.width=width
        self.height=height
        self.rtsp_url = rtsp_url
        self.max_missed_frames = max_missed_frames
        self.frame_q = queue.Queue(maxsize = max_buffer_size)
        self.container_open_wait_time = 5

        self.__init_rtsp_stream()
        # Note: we are not handling it when the initial rtsp stream fails to open here. 
        # The decoder thread will try to open it until it succeeds (or the app is closed)
 
    def get(self):
        return self.__get_next_frame()
    
    def __close_rtsp_container(self):
        if self.container_open:
            self.container.close()
            self.container_open = False

    def __open_rtsp_container(self):
        if not self.container_open:
            self.container = None
            try:
                self.container = av.open(self.rtsp_url, timeout=(10,5))
                self.container_open = True
                self.container_open_wait_time = MIN_CONTAINER_OPEN_WAIT_TIME
                print(f"RTSP URL is opened: {self.container}")
            except Exception as e:
                # We back off the retries up to a max. wait time to not overload the system with too many attempts to open the stream.
                # Different exceptions should also be handled differently, e.g. you might want to signal authentication problems
                print(f"open rtsp container exception: {e}")
                time.sleep(self.container_open_wait_time)
                self.container_open_wait_time = min(self.container_open_wait_time + MIN_CONTAINER_OPEN_WAIT_TIME, MAX_CONTAINER_OPEN_WAIT_TIME)

    
    def __init_rtsp_stream(self):
        if self.initialized:
            # Signal the decode thread to exit
            self.stop_decoding = True

            # Wait for decode thread to exit
            self.decode_thread.join()

            # release any resources
            self.__close_rtsp_container()

            # empty the frame queue
            while True:
                try:
                    self.frame_q.get(block = False)
                except queue.Empty as e:
                    print("Queue is now empty")
                    break

            self.initialized = False

        # Setup the input container interface
        try:
            # Open the RTSP stream
            self.__open_rtsp_container()

            # Start the decoder thread
            self.stop_decoding = False
            self.decode_thread = threading.Thread(target = self.__decoder_worker, daemon=True)
            self.decode_thread.start()

            self.initialized = True
        except av.AVError as e:
            print(f"Failed to open stream: {e}")

    def __decoder_worker(self):
        while not self.stop_decoding:
            if self.container_open:
                try:
                    for frame in self.container.decode(video=0):
                        # print(f"decoded frame: {frame.width} x {frame.height} timestamp: {frame.time}")
                        try:
                            # converting the received image into a format that cv2 can handle
                            # change this code to what is needed by the application
                            normalized_frame = frame.reformat(width=self.width, height=self.height)
                            cv2_image = normalized_frame.to_ndarray(format='bgr24')
                            self.frame_q.put(cv2_image, block = False)
                        except queue.Full as e:
                            print("Dropping frame")
                        except Exception as e:
                            print(f"Unexpected exception: {e}")

                # Different exceptions should also be handled differently, e.g. 
                # * you might want to signal authentication problems
                # * prolonged disconnects should be signaled
                except av.AVError as e:
                    print(f"Failed to get frame in decoder thread: {e}")
                    print(f"Let's try to restart the RTSP container")
                    self.__close_rtsp_container()
                    self.__open_rtsp_container()
                except Exception as e:
                    print(f"Unexpected exception in decoder thread: {e}")
                    print(f"Let's try to restart the container in this case also.")
                    self.__close_rtsp_container()
                    self.__open_rtsp_container()
            else:
                print("Decoder thread: container not open yet. Let's try again to pen the RTSP stream.")
                self.__open_rtsp_container()


    def __get_next_frame(self):
        next_frame = None
        while not self.stop_decoding:
            try:
                next_frame =  self.frame_q.get(timeout=1)
                self.no_frame_count = 0
                break
            except queue.Empty:
                # Not doing anything about an empty queue, just retrying
                # It might make sense to add a back off mechanism, e.g. increase the queue get timeout
                pass
            except Exception as e:
                print(f"Get next frame, unexpected exception: {e}")

        return next_frame
    
class PyAvApp:
    def __init__(self, rtsp_url, stream_max_buffer_size = 20, stream_max_missed_frames = 10, stream_width=800, stream_height=600):
        print("PyAvApp init.")
        self.video_in = PyAvStream(
            rtsp_url, 
            max_buffer_size=stream_max_buffer_size,
            max_missed_frames=stream_max_missed_frames,
            width=stream_width,
            height=stream_height)